package presentation;

/**
 * @author Alberto Iachini
 */

import java.awt.GridBagLayout;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JLabel;
import java.awt.GridBagConstraints;
import javax.swing.JButton;
import javax.swing.JList;
import javax.swing.DefaultListModel;

import java.awt.Insets;
import java.util.ArrayList;
import java.util.List;
import org.jgraph.JGraph;

import presentation.graph.GraphFactory;
import presentation.util.Exporter;

import abstraction.ReadFile;
import abstraction.Service;

import javax.swing.JScrollPane;
import javax.swing.filechooser.FileNameExtensionFilter;

import abstraction.AvailableServices;

import java.io.File;
import java.awt.Dimension;

/**
 * This class is the panel used to show available services and add and remove these services
 *
 */
public class AvailablePanel extends JPanel {

	private static final long serialVersionUID = -9064608565705075447L;
	private OrchestratorFrame owner;
	private JLabel availableLabel = null;
	private JLabel statusLabel = null;
	
	private JButton addServicesButton = null;
	private JButton removeServiceButton = null;
	private JButton removeAllButton = null;
	private JButton saveToImageButton = null;
	private JButton saveToDotButton = null;
	
	
	private JScrollPane graphScrollPane = null;
	private JScrollPane listScrollPane = null;
	
	private JList servicesList = null;
	private DefaultListModel servicesListModel = null;
	
	private List<JGraph> availableGraphs;
	
	private boolean availableLoaded;

	private AvailableServices services;

	/**
	 * Returns the available services
	 * @return the object that contains the available services
	 */
	public AvailableServices getServices() {
		return services;
	}

	/**
	 * This is the default constructor
	 */
	public AvailablePanel(OrchestratorFrame frame) {
		super();
		owner = frame;
		services = new AvailableServices();
		servicesListModel = new DefaultListModel();
		initialize();
	}

	/**
	 * This method initializes this
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints12 = new GridBagConstraints();
		gridBagConstraints12.fill = GridBagConstraints.BOTH;
		gridBagConstraints12.gridy = 6;
		gridBagConstraints12.weightx = 0.0;
		gridBagConstraints12.weighty = 1.0;
		gridBagConstraints12.ipadx = 10;
		gridBagConstraints12.gridheight = 3;
		gridBagConstraints12.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints12.gridwidth = 1;
		gridBagConstraints12.anchor = GridBagConstraints.WEST;
		gridBagConstraints12.gridx = 0;
		GridBagConstraints gridBagConstraints31 = new GridBagConstraints();
		gridBagConstraints31.gridy = 1;
		gridBagConstraints31.fill = GridBagConstraints.BOTH;
		gridBagConstraints31.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints31.gridheight = 9;
		gridBagConstraints31.weightx = 10.0;
		gridBagConstraints31.weighty = 10.0;
		gridBagConstraints31.gridx = 1;
		GridBagConstraints gridBagConstraints11 = new GridBagConstraints();
		gridBagConstraints11.fill = GridBagConstraints.NONE;
		gridBagConstraints11.gridy = 1;
		gridBagConstraints11.weighty = 1.0;
		gridBagConstraints11.gridx = 2;
		GridBagConstraints gridBagConstraints21 = new GridBagConstraints();
		gridBagConstraints21.gridx = 1;
		gridBagConstraints21.gridwidth = 1;
		gridBagConstraints21.anchor = GridBagConstraints.EAST;
		gridBagConstraints21.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints21.fill = GridBagConstraints.HORIZONTAL;
		gridBagConstraints21.weightx = 10.0;
		gridBagConstraints21.gridy = 0;
		statusLabel = new JLabel();
		statusLabel.setText("No available service loaded");
		GridBagConstraints gridBagConstraints5 = new GridBagConstraints();
		gridBagConstraints5.gridx = 0;
		gridBagConstraints5.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints5.ipadx = 38;
		gridBagConstraints5.anchor = GridBagConstraints.WEST;
		gridBagConstraints5.gridy = 5;
		GridBagConstraints gridBagConstraints4 = new GridBagConstraints();
		gridBagConstraints4.gridx = 0;
		gridBagConstraints4.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints4.ipadx = 49;
		gridBagConstraints4.anchor = GridBagConstraints.WEST;
		gridBagConstraints4.gridy = 4;
		GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
		gridBagConstraints3.gridx = 0;
		gridBagConstraints3.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints3.ipadx = 16;
		gridBagConstraints3.anchor = GridBagConstraints.WEST;
		gridBagConstraints3.gridy = 3;
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.ipadx = 40;
		gridBagConstraints2.anchor = GridBagConstraints.WEST;
		gridBagConstraints2.gridy = 2;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.ipadx = 29;
		gridBagConstraints1.anchor = GridBagConstraints.WEST;
		gridBagConstraints1.gridy = 1;
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.ipadx = 0;
		gridBagConstraints.ipady = 0;
		gridBagConstraints.anchor = GridBagConstraints.WEST;
		gridBagConstraints.gridy = 0;
		availableLabel = new JLabel();
		availableLabel.setText("Available Services");
		this.setLayout(new GridBagLayout());
		this.setSize(new Dimension(360, 317));
		this.add(availableLabel, gridBagConstraints);
		this.add(getAddServicesButton(), gridBagConstraints1);
		this.add(getRemoveServiceButton(), gridBagConstraints2);
		this.add(getRemoveAllButton(), gridBagConstraints3);
		this.add(getSaveToImageButton(), gridBagConstraints4);
		this.add(getSaveToDotButton(), gridBagConstraints5);
		this.add(statusLabel, gridBagConstraints21);
		this.add(getGraphScrollPane(), gridBagConstraints31);
		this.add(getListScrollPane(), gridBagConstraints12);
	}

	/**
	 * This method initializes addServicesButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getAddServicesButton() {
		if (addServicesButton == null) {
			addServicesButton = new JButton();
			addServicesButton.setText("Add New Services");
			addServicesButton.setToolTipText("Add one or more available service(s)");
			addServicesButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					
					/*
					 * - get Files with JFileChooser
					 * - check if List availableGraph is null or not
					 * - for each file
					 *  	 - create availableService with ReadFile
					 *  	 - add service to community
					 * - set the flag availableLoaded to true
					 */
					File[] files = openFiles();
					
					int numFiles = files.length;
					if(availableGraphs==null)
						availableGraphs = new ArrayList<JGraph>(numFiles);
					
					int loadedFiles = 0;
					for(int i=0; i<numFiles; i++)
					{
						if (files[i].exists()) {
							ReadFile.setFileName(files[i].getAbsolutePath());
							Service serv = null;
							try {
								String target = ReadFile.readFile();
								serv = new Service(target);
								String name = files[i].getName();
								servicesListModel.addElement(name);

								services.addService(serv);

								availableGraphs.add(null);
								loadedFiles++;
							} catch (Exception e1) {
								String error = e1.getMessage();
								JOptionPane.showMessageDialog(null, error,"Error", JOptionPane.ERROR_MESSAGE);
							}
						}
					}
					int size = services.sizeCommunity();
					statusLabel.setText("Service(s): "+ size +" - New service(s) Loaded: "+loadedFiles);
					if(loadedFiles>0)
						availableLoaded = true;
				}
			});
		}
		return addServicesButton;
	}

	/**
	 * This method initializes removeServiceButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getRemoveServiceButton() {
		if (removeServiceButton == null) {
			removeServiceButton = new JButton();
			removeServiceButton.setText("Remove Service");
			removeServiceButton.setToolTipText("Remove selected service");
			removeServiceButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					/*
					 *  get index from serviceList
					 *  delete the selected service
					 *  if community or model is empty then set false the flag availableLoaded 
					 */
					int index = servicesList.getSelectedIndex();
					if (availableLoaded==true && index > -1) {
						
						String name = (String) servicesListModel.get(index);
						services.deleteService(index);
						availableGraphs.remove(index);
						servicesListModel.remove(index);
						graphScrollPane.setViewportView(null);
						statusLabel.setText("Deleted Service "+name);
						if (servicesListModel.isEmpty())
						{
							availableLoaded = false;
							statusLabel.setText("No available service loaded");
						}
					}
				}
			});
		}
		return removeServiceButton;
	}

	/**
	 * This method initializes removeAllButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getRemoveAllButton() {
		if (removeAllButton == null) {
			removeAllButton = new JButton();
			removeAllButton.setText("Remove All Services");
			removeAllButton.setToolTipText("Remove all services");
			removeAllButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					// remove all services
					
					if (availableLoaded==true) {
						int lenght = servicesListModel.size();
						for (int i = 0; i < lenght; i++) {
							services.deleteService(0);
							availableGraphs.remove(0);
							servicesListModel.remove(0);
							
						}
						graphScrollPane.setViewportView(null);
						availableLoaded = false;
						statusLabel.setText("No available service loaded");
					}
				}
			});
		}
		return removeAllButton;
	}

	/**
	 * This method initializes saveToImageButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToImageButton() {
		if (saveToImageButton == null) {
			saveToImageButton = new JButton();
			saveToImageButton.setText("Save To Image");
			saveToImageButton.setToolTipText("Save the graph in a jpeg file");
			saveToImageButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					int index = servicesList.getSelectedIndex();
					if (index > -1) {
						File file = saveImageFile();
						if (file != null)
							Exporter.exportToImage(file, availableGraphs.get(index));
					}
				}
			});
		}
		return saveToImageButton;
	}

	/**
	 * This method initializes saveToDotButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToDotButton() {
		if (saveToDotButton == null) {
			saveToDotButton = new JButton();
			saveToDotButton.setText("Save To DOT File");
			saveToDotButton.setToolTipText("Save the graph in DOT format");
			saveToDotButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					int index = servicesList.getSelectedIndex();
					if (index > -1) {
						File file = saveDotFile();
						if (file != null)
							Exporter.exportToDot(file, services.get(index));
					}
				}
			});
		}
		return saveToDotButton;
	}

	/**
	 * This method initializes graphScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getGraphScrollPane() {
		if (graphScrollPane == null) {
			graphScrollPane = new JScrollPane();
			graphScrollPane.setViewportView(null);
		}
		return graphScrollPane;
	}

	/**
	 * This method initializes listScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getListScrollPane() {
		if (listScrollPane == null) {
			listScrollPane = new JScrollPane();
			listScrollPane.setViewportView(getServicesList());
		}
		return listScrollPane;
	}

	/**
	 * This method initializes servicesList	
	 * 	
	 * @return javax.swing.JList	
	 */
	private JList getServicesList() {
		if (servicesList == null) {
			
			servicesList = new JList(servicesListModel);
			servicesList.setToolTipText("Choose an Available Service to show");
			servicesList
					.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
						public void valueChanged(javax.swing.event.ListSelectionEvent e) {
							/*
							 * - show the selected JGraph graph 
							 * - if graph is null
							 *      - create the graph with GraphFactory
							 */
							int index = servicesList.getSelectedIndex();
							if (index>-1) {
								JGraph graph = availableGraphs.get(index);
								if (graph == null) {
									graph = (new GraphFactory(services.get(index))).getGraph();
									availableGraphs.set(index, graph);
								}
								graphScrollPane.setViewportView(graph);
								statusLabel.setText("Available Service: "+ services.get(index).getName());
							}
						}
					});
		}
		return servicesList;
	}
	
	/**
	 * Open a JFileChooserMultiselection to load the available service files
	 * 
	 * @return an array of files
	 */
	private File[] openFiles()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("Graph File", "txt");
		fc.setFileFilter(filter);
		fc.setMultiSelectionEnabled(true);
		fc.showOpenDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFiles();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into jpeg file
	 * 
	 * @return the selected file
	 */
	private File saveImageFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("JPEG File", "jpeg");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into dot file
	 * 
	 * @return the selected file
	 */
	private File saveDotFile()
	{
		JFileChooser fc = new JFileChooser();
		
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("DOT File", "dot");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		owner.setLastFolder(fc.getCurrentDirectory());
		return fc.getSelectedFile();
	}
	
	/**
	 * This method check if at least one service is loaded
	 * 
	 * @return true if one or more service are loades
	 */
	public boolean isAvailableLoaded() {
		return availableLoaded;
	}
	
	/**
	 * Enable or disable the add\remove buttons
	 * 
	 * @param flag if true enables the buttons
	 */
	public void setEnableButtons(boolean flag)
	{
		this.addServicesButton.setEnabled(flag);
		this.removeServiceButton.setEnabled(flag);
		this.removeAllButton.setEnabled(flag);
	}

}  //  @jve:decl-index=0:visual-constraint="24,11"
